<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gainup - Yearly Budget Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.datatables.net/2.1.8/css/dataTables.dataTables.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/buttons/3.0.1/css/buttons.dataTables.min.css">


	<script src="https://code.jquery.com/jquery-3.7.1.js"></script>
<script src="https://cdn.datatables.net/2.1.8/js/dataTables.min.js"></script>

<script src="https://cdn.datatables.net/buttons/3.0.1/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/3.0.1/js/buttons.html5.min.js"></script>

<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>

    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        html, body { height: 100%; }
        body { background: #f8f9fa; font-family: 'Segoe UI', Roboto, -apple-system, BlinkMacSystemFont, Arial; font-size: 14px; color: #12263a; }
        
        .viewport { 
            display: flex; 
            flex-direction: column; 
            height: 100vh; 
            overflow: hidden;
            padding: 4px;
            gap: 4px;
        }
        
        .budget-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 16px;
            border-radius: 6px;
            margin-bottom: 4px;
            flex-shrink: 0;
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 16px;
        }
        
        .header-left {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        
        .budget-header h4 { 
            font-size: 1.25rem; 
            font-weight: 700; 
            margin: 0;
            letter-spacing: 0.2px;
            color: #fff;
        }
        
        .budget-header small { 
            opacity: 0.95; 
            font-size: 0.85rem;
            color: rgba(255,255,255,0.92);
        }
        
        .header-controls {
            display: flex;
            gap: 8px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .year-select, .header-select {
            padding: 5px 8px;
            border: none;
            border-radius: 4px;
            font-size: 0.8rem;
            background: rgba(255,255,255,0.9);
            color: #333;
        }
        
        .btn-header {
            padding: 5px 10px;
            font-size: 0.75rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
            background: rgba(255,255,255,0.2);
            color: white;
        }
        
        .btn-header:hover {
            background: rgba(255,255,255,0.3);
        }
        
        .content-grid {
            display: grid;
            grid-template-columns: 1.8fr 0.9fr 0.8fr;
            gap: 4px;
            flex-shrink: 0;
            height: auto;
        }
        
        .card-section { 
            background: white; 
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 10px;
            overflow-y: auto;
            display: flex;
            flex-direction: column;
            box-shadow: 0 1px 3px rgba(16,24,40,0.04);
        }
        
        .card-section h6 { 
            font-size: 0.9rem; 
            font-weight: 600; 
            margin: 0 0 8px 0; 
            color: #222;
            flex-shrink: 0;
        }
        
        .chart-box { 
            position: relative; 
            height: 120px;
            width: 100%;
            flex-shrink: 0;
        }
        
        .grid-table {
            width: 100%;
            font-size: 0.8rem;
            border-collapse: collapse;
        }
        
        .grid-table thead {
            background: #f8f9fa;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .grid-table th {
            padding: 10px 8px;
            font-weight: 700;
            text-align: left;
            border-bottom: 2px solid #dce6ea;
            font-size: 0.85rem;
            color: #0f1724;
            background: #fafcfe;
        }
        
        .grid-table td {
            padding: 10px 8px;
            border-bottom: 1px solid #f3f6f7;
            color: #12263a;
            font-size: 0.9rem;
        }
        
        .kpi-summary {
            display: flex;
            flex-direction: column;
            gap: 6px;
            flex: 1;
            min-height: 0;
            overflow-y: auto;
        }
        
        .kpi-row {
            display: flex;
            justify-content: space-between;
            padding: 4px 4px;
            border-bottom: 1px solid #f0f0f0;
            font-size: 0.8rem;
        }
        
        .kpi-row:last-child {
            border-bottom: none;
        }
        
        .kpi-label { 
            color: #666; 
            font-weight: 500;
        }
        
        .kpi-value { 
            color: #333;
            font-weight: 600;
        }
        
        .controls-section {
            display: flex;
            flex-direction: column;
            gap: 6px;
        }
        
        .control-group {
            display: flex;
            flex-direction: column;
        }
        
        .control-group label {
            font-size: 0.75rem;
            color: #666;
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .control-group select {
            font-size: 0.8rem;
            padding: 4px 6px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .btn-control {
            padding: 5px 10px;
            font-size: 0.75rem;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 500;
        }
        
        .btn-primary-sm {
            background: #667eea;
            color: white;
        }
        
        .btn-primary-sm:hover {
            background: #5568d3;
        }
        
        .btn-secondary-sm {
            background: #e0e0e0;
            color: #333;
        }
        
        .btn-secondary-sm:hover {
            background: #d0d0d0;
        }
        
        .left-charts {
            display: contents;
        }
        
        .chart-card {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 6px 8px;
            display: flex;
            flex-direction: column;
            height: fit-content;
            box-shadow: 0 1px 2px rgba(16,24,40,0.04);
        }
        
        .chart-card h6 {
            font-size: 0.9rem;
            font-weight: 700;
            margin: 0 0 4px 0;
            color: #0f1724;
            flex-shrink: 0;
        }
        
        .right-controls {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 6px 8px;
            display: flex;
            flex-direction: column;
            box-shadow: 0 1px 3px rgba(16,24,40,0.03);
        }
        
        .full-grid {
            background: white;
            border: 1px solid #d1d9de;
            border-radius: 8px;
            padding: 8px;
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            box-shadow: 0 1px 4px rgba(16,24,40,0.04);
        }
        
        .full-grid h6 {
            font-size: 0.9rem;
            font-weight: 600;
            margin: 0 0 8px 0;
            color: #222;
            flex-shrink: 0;
        }
        
        .grid-scroll {
            overflow-y: auto;
            flex: 1;
            min-height: 0;
        }
        
        @media (max-width: 1400px) {
            .content-grid {
                grid-template-columns: 1fr 1fr;
            }
            .right-controls {
                grid-column: 1 / -1;
            }
        }
        
        @media (max-width: 1024px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
        }
		
		/* Make toolbar row flex */
.dataTables_wrapper .dt-layout-row {
    display: flex !important;
    align-items: center !important;
}

/* Push everything to the right */
.dataTables_wrapper .dt-layout-cell {
    display: flex;
    justify-content: flex-end !important;
    gap: 10px;
}

/* Smaller button */
.dt-button {
    padding: 4px 10px !important;
    font-size: 12px !important;
    border-radius: 4px !important;
}

/* Smaller Search input */
.dataTables_filter input,
.dt-search input {
    height: 28px !important;
    font-size: 12px !important;
}
/* Make all table cells stay in one line */
#budgetgrid td, 
#budgetgrid th {
    white-space: nowrap !important;
    overflow: hidden;
    text-overflow: ellipsis;   /* show ... when too long */
    vertical-align: middle !important;
    padding-top: 6px !important;
    padding-bottom: 6px !important;
}

/* Make table more compact */
#budgetgrid {
    font-size: 13px;
    line-height: 1.2;
}

/* Fix header alignment */
#budgetgrid thead th {
    font-weight: 600 !important;
}
#budgetgrid td {
    padding: 4px 8px !important;
}

#budgetgrid th {
    white-space: nowrap !important;
}
    </style>
</head>
<body>
    <div class="viewport">
        <!-- Header -->
        <div class="budget-header">
            <div class="header-left">
                <h4> Budget Dashboard <span>(Annual Budget Allocation & Usage Tracking)</span></h4>
                
            </div>
            <div class="header-controls">
                <select id="departmentFilter" class="header-select">
                    <option value="">All Departments</option>
                    <!-- Populated dynamically from API data -->
                </select>
                <!--<select id="yearSelect" class="header-select">
                    <option value="2023">2023</option>
                    <option value="2024" selected>2024</option>
                    <option value="2025">2025</option>
                </select>
                <button class="btn-header" onclick="exportData()">📥 Export</button>-->
                <button class="btn-header" onclick="resetFilters()">🔄 Reset</button>
            </div>
        </div>

        <!-- Main Content Grid (3 columns: Chart1, Chart2, Summary) -->
        <div class="content-grid">
            <div class="left-charts">
                <!-- Chart 1 -->
                <div class="chart-card">
                    <h6>📊 As On Month vs Used Budget</h6>
                    <div class="chart-box">
                        <canvas id="asonBarChart"></canvas>
                    </div>
                </div>

                <!-- Chart 2 -->
                <div class="chart-card">
                    <h6>🎯 Total Budget Allocation</h6>
                    <div class="chart-box">
                        <canvas id="totalDoughnutChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Summary -->
            <div class="right-controls">
                <h6>📈 Summary</h6>
                <div class="kpi-summary" id="quickSummary">
                    <!-- Populated by JS -->
                </div>
            </div>
        </div>

        <!-- Full Width Grid -->
        <div class="full-grid">
            <h6>📋 Department Budget Details <span style="color:red">(Value in thousands) </span></h6>
            <div class="grid-scroll">
                <table class="grid-table" id="budgetgrid">
                    <thead>
                        <tr>
							<th>Company</th>
							<th>Project ID</th>
                            <th>Division</th>
                            <th>Project</th>
                            <th>Activity</th>
                            <!--<th>Description</th>-->
                            <th>Budget<br> Incharge</th>                            
                            <th>Total<br> Budget (FY)</th>
                            <th>As On <br>Month (A)</th>
                            <th>Used <br>Budget (B)</th>
                            <th>Pending <br>Budget (A-B)</th>
							<th>Budget <br>Overrun</th>
                        </tr>
                    </thead>
                    <tbody id="budgetGrid">
                        <!-- Populated by JS -->
                    </tbody>
					
                </table>
            </div>
        </div>
    </div>

    <script>
        
        let budgetData = [];
        let currentData = [];
        let asonChart, doughnutChart;

        function formatMoney(value) {
            let formatted;
			//if (value >= 100000) {
			//	formatted = (value / 100000).toFixed(1); // Lakhs
			//	formatted = formatted.endsWith('.0') ? formatted.slice(0, -2) : formatted;
			//	return '₹' + formatted + 'L';
			//} else {
				formatted = (value / 1000).toFixed(1); // Thousands
				formatted = formatted.endsWith('.0') ? formatted.slice(0, -2) : formatted;
				return formatted;
			//}
        }

        function transformApiData(apiResponse) {
            // Group API data by Project/Department
            const grouped = {};
            
            if (apiResponse.data && Array.isArray(apiResponse.data)) {
                apiResponse.data.forEach(item => {
                    const key = item.Project || 'Unknown';
                    if (!grouped[key]) {
                        grouped[key] = {
                            dept: item.Project || 'Unknown',
                            division: item.Division || 'General',
                            yearBudget: 0,
                            ason: 0,
                            used: 0,
                            items: []
                        };
                    }
                    
                    grouped[key].yearBudget += parseFloat(item.Total_Budget || 0);
                    grouped[key].ason += parseFloat(item.Budget_As_On_Month || 0);
                    grouped[key].used += parseFloat(item.Used_Budget || 0);
                    grouped[key].items.push(item);
                });
            }
            
            return Object.values(grouped);
        }

        function populateDepartmentFilter(departments) {
            const select = document.getElementById('departmentFilter');
            const currentOptions = select.innerHTML;
            
            departments.forEach(dept => {
                const option = document.createElement('option');
                option.value = dept.dept;
                option.textContent = dept.dept;
                select.appendChild(option);
            });
        }

        function initCharts() {
            const barCtx = document.getElementById('asonBarChart');
            const doughnutCtx = document.getElementById('totalDoughnutChart');

            if (asonChart) asonChart.destroy();
            if (doughnutChart) doughnutChart.destroy();

            // Prepare data from individual items for bar chart
            const projectData = {};
            currentData.forEach(dept => {
                if (dept.items && Array.isArray(dept.items)) {
                    dept.items.forEach(item => {
						const projectName = (item.Project || '-').substring(0, 10);
                        if (!projectData[projectName]) {
                            projectData[projectName] = {
                                budgetAsOn: 0,
                                usedBudget: 0,
                                pendingBudget: 0
                            };
                        }
                        projectData[projectName].budgetAsOn += parseFloat(item.Budget_As_On_Month || 0);
                        projectData[projectName].usedBudget += parseFloat(item.Used_Budget || 0);
                        projectData[projectName].pendingBudget += parseFloat(item.Pending_Budget || 0);
                    });
                }
            });

            const labels = Object.keys(projectData);
            const budgetAsOnValues = labels.map(label => projectData[label].budgetAsOn);
            const usedBudgetValues = labels.map(label => projectData[label].usedBudget);
            const pendingBudgetValues = labels.map(label => projectData[label].pendingBudget);

            // Bar Chart - Budget_As_On_Month vs Used_Budget vs Pending_Budget (Vertical)
            asonChart = new Chart(barCtx.getContext('2d'), {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [
                        { 
                            label: 'Budget As On Month', 
                            data: budgetAsOnValues, 
                            backgroundColor: '#667eea',
                            borderRadius: 4,
                            borderWidth: 1,
                            borderColor: '#5568d3'
                        },
                        { 
                            label: 'Used Budget', 
                            data: usedBudgetValues, 
                            backgroundColor: '#ff6b6b',
                            borderRadius: 4,
                            borderWidth: 1,
                            borderColor: '#ee5a52'
                        },
                        { 
                            label: 'Pending Budget', 
                            data: pendingBudgetValues, 
                            backgroundColor: '#ffd93d',
                            borderRadius: 4,
                            borderWidth: 1,
                            borderColor: '#f5c933'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { 
                            position: 'top', 
                            labels: { 
                                font: { size: 10 }, 
                                padding: 8,
                                usePointStyle: true
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ₹' + (context.parsed.y / 100000).toFixed(1) + 'L';
                                }
                            }
                        }
                    },
                    scales: {
                        y: { 
                            beginAtZero: true,
                            stacked: false,
                            ticks: { 
                                font: { size: 9 }, 
                                callback: v => '₹' + (v / 100000).toFixed(1) + 'L'
                            }
                        },
                        x: { 
                            ticks: { font: { size: 9 } },
                            stacked: false
                        }
                    }
                }
            });

            // Doughnut Chart
            const totals = currentData.reduce((s, d) => ({
                budget: s.budget + d.yearBudget,
                used: s.used + d.used
            }), { budget: 0, used: 0 });

            doughnutChart = new Chart(doughnutCtx.getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: ['Used', 'Remaining'],
                    datasets: [{
                        data: [totals.used, totals.budget - totals.used],
                        backgroundColor: ['#ff6b6b', '#4ecdc4'],
                        borderColor: '#fff',
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'bottom', labels: { font: { size: 10 }, padding: 6 } },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return context.label + ': ₹' + (context.parsed / 100000).toFixed(1) + 'L';
                                }
                            }
                        }
                    }
                }
            });
        }

        function populateGrid() {
            const tbody = document.getElementById('budgetGrid');
            const rows = [];
            
            currentData.forEach(dept => {
                if (dept.items && Array.isArray(dept.items)) {
                    dept.items.forEach(item => {
                        // Truncate fields to specified character limits
                        const company = (item.Company || '-').substring(0, 10);
                        const activity = (item.Proj_Activity_Name || '-').substring(0, 15);
                        const description = (item.Description || '-').substring(0, 15);
                        const employee = (item.Employee || '-').substring(0, 10);
						const Projecttxt = (item.Project || '-').substring(0, 10);
                        
                        rows.push(`<tr>
                            <td title="${item.Company || '-'}">${company}</td>
							<td title="${item.Order_No || '-'}">${item.Order_No}</td>
                            <td>${item.Division || '-'}</td>
							<td title="${item.Project || '-'}">${Projecttxt}</td>
                            <td title="${item.Proj_Activity_Name || '-'}">${activity}</td>
                            <!--<td title="${item.Description || '-'}">${description}</td>-->
                            <td title="${item.Employee || '-'}">${employee}</td>                            
                            <td><strong>${formatMoney(parseFloat(item.Total_Budget || 0))}</strong></td>
                            <td>${formatMoney(parseFloat(item.Budget_As_On_Month || 0))}</td>
                            <td><strong>${formatMoney(parseFloat(item.Used_Budget || 0))}</strong></td>
                            <td>${formatMoney(parseFloat(item.Pending_Budget || 0))}</td>
							<td></td>
                        </tr>`);
                    });
                }
            });
            
            tbody.innerHTML = rows.join('');
        }

        function updateSummary() {
            let totals = {
                totalBudget: 0,
                asonBudget: 0,
                usedBudget: 0,
                pendingBudget: 0
            };

            currentData.forEach(dept => {
                if (dept.items && Array.isArray(dept.items)) {
                    dept.items.forEach(item => {
                        totals.totalBudget += parseFloat(item.Total_Budget || 0);
                        totals.asonBudget += parseFloat(item.Budget_As_On_Month || 0);
                        totals.usedBudget += parseFloat(item.Used_Budget || 0);
                        totals.pendingBudget += parseFloat(item.Pending_Budget || 0);
                    });
                }
            });

            document.getElementById('quickSummary').innerHTML = `
                <div class="kpi-row">
                    <span class="kpi-label">Total Budget (FY)</span>
                    <span class="kpi-value">${formatMoney(totals.totalBudget)}</span>
                </div>
                <div class="kpi-row">
                    <span class="kpi-label">As On Month</span>
                    <span class="kpi-value">${formatMoney(totals.asonBudget)}</span>
                </div>
                <div class="kpi-row">
                    <span class="kpi-label">Used Budget</span>
                    <span class="kpi-value">${formatMoney(totals.usedBudget)}</span>
                </div>
                <div class="kpi-row">
                    <span class="kpi-label">Pending Budget</span>
                    <span class="kpi-value">${formatMoney(totals.pendingBudget)}</span>
                </div>
            `;
        }

        function exportData() {
            const csv = 'Department,Division,Yearly Budget,Ason Value,Used,Remaining\n' +
                currentData.map(r => `${r.dept},${r.division},${r.yearBudget},${r.ason},${r.used},${r.yearBudget - r.used}`).join('\n');
            
            const blob = new Blob([csv], { type: 'text/csv' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'budget_export.csv';
            a.click();
            URL.revokeObjectURL(url);
        }

        function resetFilters() {
            document.getElementById('departmentFilter').value = '';
            currentData = [...budgetData];
            populateGrid();
            updateSummary();
            initCharts();
        }

        document.addEventListener('DOMContentLoaded', function() {
            // Fetch budget details from API
            
            // Get current date in dd-MMM-yyyy format
            function getCurrentDateFormatted() {
                const today = new Date();
                const day = String(today.getDate()).padStart(2, '0');
                const monthNames = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", 
                                    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
                const month = monthNames[today.getMonth()];
                const year = today.getFullYear();
                return `${day}-${month}-${year}`;
            }

            const currentDate = getCurrentDateFormatted();

            fetch('budgetdetails?curdate='+currentDate)
                .then(response => response.json())
                .then(apiData => {
                    console.log('Budget Details Response:', apiData);
                    
                    // Transform API data to chart format
                    budgetData = transformApiData(apiData);
                    currentData = [...budgetData];
                    
                    // Populate department filter dropdown
                    populateDepartmentFilter(budgetData);
                    
                    // Initialize charts and populate grid
                    initCharts();
                    populateGrid();
					initDataTable();
                    updateSummary();
                })
                .catch(error => console.error('Error fetching budget details:', error));

            // Event listeners
            document.getElementById('departmentFilter').addEventListener('change', function() {
                currentData = this.value ? budgetData.filter(d => d.dept === this.value) : [...budgetData];
                populateGrid();
                updateSummary();
                initCharts();
            });

            document.getElementById('yearSelect').addEventListener('change', function() {
                console.log('Year changed to:', this.value);
            });
        });

		function initDataTable() {
			//DataTable.type('num', 'className', 'dt-body-right');
			//DataTable.type('num-fmt', 'className', 'dt-body-right');
			//DataTable.type('date', 'className', 'dt-body-right');
			new DataTable('#budgetgrid', {
				destroy: true,
				paging: false,
				searching: true,
				ordering: true,
				autoWidth: false,

				layout: {
					topStart: 'buttons',
					topEnd: 'search'
				},

				buttons: [
					{
						extend: 'excelHtml5',
						text: 'Excel',
						title: 'Budget_Details'
					}
				],

				/*initComplete: function () {
					this.api()
						.columns()
						.every(function () {
							let column = this;

							let select = document.createElement('select');
							select.add(new Option(''));
							column.header().replaceChildren(select);

							select.addEventListener('change', function () {
								column.search(select.value, { exact: true }).draw();
							});

							column.data().unique().sort().each(function (d) {
								select.add(new Option(d));
							});
						});
				}*/
			});
		}

    </script>
</body>
</html>
